package fractaltriangles;

import javafx.geometry.Point2D;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.paint.Color;

/**
 * This class models a triangle.
 * 
 * For this application, the triangles will
 * be equilateral and oriented so that one edge
 * is horizontal. It is therefore to label the
 * vertices southeast (lower right), north (top),
 * and southwest (lower left).
 * 
 * @author Leon Tabak
 * @version 28 March 2018
 */
public class Triangle {

    /**
     * The lower left corner of this triangle.
     */
    private final Point2D southwestVertex;
    
    /**
     * The lower right corner of this triangle.
     */
    private final Point2D southeastVertex;
    
    /**
     * The top corner of this triangle.
     */
    private final Point2D northVertex;

    /**
     * Construct an equilateral triangle whose center
     * is at a specified point and that fits within a circle
     * of specified size.
     * 
     * @param centerX is the x coordinate of the center.
     * @param centerY is the y coordinate of the center.
     * @param radius is the radius of the smallest circle
     * that can contain the triangle.
     */
    public Triangle(double centerX, double centerY, double radius) {
        double angle = Math.toRadians( 330 );
        double x = centerX + radius * Math.cos(angle);
        double y = centerY + radius * Math.sin(angle);
        this.southwestVertex = new Point2D(x, y);

        angle = Math.toRadians( 90 );
        x = centerX + radius * Math.cos(angle);
        y = centerY + radius * Math.sin(angle);
        this.southeastVertex = new Point2D(x, y);

        angle = Math.toRadians( 210 );
        x = centerX + radius * Math.cos(angle);
        y = centerY + radius * Math.sin(angle);
        this.northVertex = new Point2D(x, y);
    } // Triangle()

    /**
     * Construct a triangle with three specified
     * vertices.
     * 
     * @param se is the southeast vertex.
     * @param n is the north vertex.
     * @param sw is the southwest vertex.
     */
    public Triangle(Point2D se, Point2D n, Point2D sw) {
        this.southeastVertex = se;
        this.northVertex = n;
        this.southwestVertex = sw;
    } // Triangle( Point2D, Point2D, Point2D )

    /**
     * Draw this triangle.
     * 
     * @param gc is the GraphicsContext supplied by the caller.
     */
    public void draw(GraphicsContext gc) {
        double[] xCoordinates = new double[3];
        double[] yCoordinates = new double[3];

        xCoordinates[0] = southeastVertex.getX();
        xCoordinates[1] = northVertex.getX();
        xCoordinates[2] = southwestVertex.getX();

        yCoordinates[0] = southeastVertex.getY();
        yCoordinates[1] = northVertex.getY();
        yCoordinates[2] = southwestVertex.getY();

        gc.setStroke(Color.BLACK);

        gc.strokePolygon(xCoordinates, yCoordinates, 3);
    } // draw( GraphicsContext )

    /**
     * Construct a smaller triangle.
     * 
     * @return that part of this triangle that is
     * in the southeast corner.
     */
    public Triangle southeastTriangle() {
        Point2D se = southeastVertex;
        Point2D n = midpoint(southeastVertex, northVertex);
        Point2D sw = midpoint(southeastVertex, southwestVertex);
        return new Triangle(se, n, sw);
    } // southeastTriangle()

    /**
     * Construct a smaller triangle.
     * 
     * @return that part of this triangle that is
     * in the north corner.
     */
    public Triangle northTriangle() {
        Point2D se = midpoint(northVertex, southeastVertex);
        Point2D n = northVertex;
        Point2D sw = midpoint(northVertex, southwestVertex);
        return new Triangle(se, n, sw);
    } // northTriangle()

    /**
     * Construct a smaller triangle.
     * 
     * @return that part of this triangle that is
     * in the southwest corner.
     */
    public Triangle southwestTriangle() {
        Point2D se = midpoint(southwestVertex, southeastVertex);
        Point2D n = midpoint(southwestVertex, northVertex);
        Point2D sw = southwestVertex;
        return new Triangle(se, n, sw);
    } // southwestTriangle()

    /**
     * Divide this triangle into four smaller triangles
     * (one at the center of this triangle and one at each
     * of the corners)
     * and repeat the process with the three outer triangles
     * until depth is zero.
     * 
     * This is a recursive method. All recursive methods contain
     * an if statement, because some decision must be made to
     * stop recursion. Every recursive method contains one or
     * more calls to itself---the name of the method that we are
     * defining occurs within the definition.
     * 
     * @param gc is the GraphicsContext supplied by the caller.
     * @param depth is a non-negative count of how many more
     * times to subdivide the triangle.
     */
    public void subdivide(GraphicsContext gc, int depth) {
        if (depth < 1) {
            // draw a triangle when it is small enough---that
            // is, when we have decided that the time has come
            // to quit dividing the triangles into smaller
            // triangles
            draw(gc);
        } // if
        else {
            // 3 recursive calls to subdivide()---the
            // method calls itself (like for loops and
            // while loops, this is a way of repeating 
            // an action)
            southeastTriangle().subdivide(gc, depth - 1);
            northTriangle().subdivide(gc, depth - 1);
            southwestTriangle().subdivide(gc, depth - 1);
        } // else    
    } // subdivide( int )

    /**
     * Construct the point that lies halfway between
     * two given points.
     * 
     * In this application, this method finds the 
     * midpoint of one side of a triangle.
     * 
     * @param p0 is one endpoint of a line segment.
     * @param p1 is the other endpoint of a line segment.
     * @return the point midway between p0 and p1.
     */
    public Point2D midpoint(Point2D p0, Point2D p1) {
        double x = (p0.getX() + p1.getX()) / 2;
        double y = (p0.getY() + p1.getY()) / 2;
        return new Point2D(x, y);
    } // midpoint( Point2D, Point2D )

} // Triangle
