package fractal;

// Tell the compiler where to find the
// definitions of the classes that this
// program uses.
import javafx.application.Application;
import javafx.scene.Scene;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.image.PixelWriter;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.stage.Stage;

/**
 * This is an experiment with bit-mapped graphics.
 *
 * The Bitmap class inherits from the Application
 * class (that is what "extends" means), so it gets
 * all of the instance variables and public methods
 * that are defined in the Application class "for free."
 * 
 * @author Your Name
 * @version 22 March 2018
 */
public class Bitmap extends Application {

    // XMIN, XMAX, YMIN, YMAX define the rectangular
    // part of the picture that the program will draw
    // (to get whole picture:
    //     XMIN = -1
    //     XMAX = +1
    //     YMIN = -1
    //     YMAX = +1
    // Try other values!
    private static final double XMIN = -0.75;
    private static final double XMAX = -0.25;
    private static final double YMIN =  0.25;
    private static final double YMAX =  0.75;
    
    // change WIDTH and HEIGHT to change
    // size of the picture
    private static final int WIDTH = 512;
    private static final int HEIGHT = 512;
    
    // a higher value for MAX_ITERATIONS will
    // produce a more accurate image of the set
    // but a slower computation
    private static final int MAX_ITERATIONS = 64;

    private final Palette palette = 
            new Palette(MAX_ITERATIONS);
    
    @Override
    public void start(Stage stage) {
        stage.setTitle("Fractal Image");

        // Create a canvas on which to draw a picture.
        Canvas canvas = new Canvas(WIDTH, HEIGHT);
        // Get that piece of the canvas that can be
        // used for drawing.
        GraphicsContext gc = canvas.getGraphicsContext2D();

        // Create a container in which to 
        // assemble the visible pieces of the
        // program (for this program, there is
        // only one piece---the canvas).
        Pane root = new Pane();

        root.getChildren().add(canvas);

        PixelWriter pw = gc.getPixelWriter();
        
        // The canvas is a rectangular array
        // of pixels. Scan that array from
        // top to bottom and from left to right.
        for (int i = 0; i < HEIGHT; i++) {
            for (int j = 0; j < WIDTH; j++) {

                // Compute a color for each pixel.
                Color c = computeColor(i, j);
                pw.setColor(j, i, c);
            } // for
        } // for

        // Finish putting the pieces together.
        Scene scene = new Scene(root);
        stage.setScene(scene);

        // Display the completed picture!
        stage.show();
    } // start( Stage )

    // Produce an index in the range 0 to size - 1,
    // where size is the number of colors in the
    // palette.
    private int computeColorIndex(int row, int column) {
        Point p = new Point(row, column, HEIGHT, WIDTH);

        double x = p.getX();
        double y = p.getY();
        
        // These next 2 statements have the
        // effect of zooming in on the image.
        x = XMIN + (XMAX - XMIN) * (x + 1)/2;
        y = YMIN + (YMAX - YMIN) * (y + 1)/2;

        // To produce an image of one of the Julia
        // sets rather than an image of the Mandelbrot
        // set, give z a real component equal to x and an
        // imaginary component equal to y, and then
        // give the real and imaginary components of c
        // constant values that you choose. (Try several
        // or many until you get a picture that you like.)
        Complex z = new Complex(0, 0);
        Complex c = new Complex(x, y);
        
        // Instead of comparing the magnitude
        // of z to 2, compare the square of the
        // magnitude of z to 4.
        // (This avoids the needs to compute a square
        // root and makes the computation faster.)
        int count = 0;
        while (z.magnitudeSquared() < 4
                && count < MAX_ITERATIONS - 1) {
            
            // compute z = z * z + c
            z = z.multiply(z);
            z = z.add(c);
            count++;
        } // while
        
        return count;
    } // computeColorIndex( int , int )

    // Use an index to select a color for a pixel.
    public Color computeColor( int row, int column ) {
        int i = computeColorIndex( row, column );
        return this.palette.getColor(i);
    } // computeColor( int, int )
    
    // Execution of the program begins in
    // the main() method.
    public static void main(String[] args) {
        launch(args);
    } // main( String[] )

} // Bitmap
