package news;

import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.io.SyndFeedInput;
import com.sun.syndication.io.XmlReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import javafx.application.Application;
import javafx.scene.Scene;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuBar;
import javafx.scene.control.MenuItem;
import javafx.scene.control.TextArea;
import javafx.scene.layout.BorderPane;
import javafx.scene.text.Text;
import javafx.scene.text.TextFlow;
import javafx.stage.Stage;

public class NewsReader extends Application {

    private final int WIDTH = 768;
    private final int HEIGHT = 512;

    private final List<Story> stories = new ArrayList<>();
    private final Random rng = new Random();

    private final String DEUTSCHE_WELLE_NAME = "Deutsche Welle";
    private final String DEUTSCHE_WELLE_URL
            = "http://rss.dw.com/xml/rss-de-deutschlandentdecken";

    private final String GOETHE_INSTITUT_NAME = "Goethe Institut";
    private final String GOETHE_INSTITUT_URL
            = "https://www.goethe.de/de/feed/kul.rss";

    private final String HELMHOLTZ_ASSOC_NAME = "Helmholtz Association";
    private final String HELMHOLTZ_ASSOC_URL
            = "https://www.helmholtz.de/aktuell/"
            + "presse_und_medien/mediathek/rss_feeds/technologie/";

    private final String SUEDDEUTSCHE_ZEITUNG_NAME = "Süddeutsche Zeitung";
    private final String SUEDDEUTSCHE_ZEITUNG_URL
            = "http://rss.sueddeutsche.de/rss/Topthemen";

    private final TextFlow textFlow = new TextFlow();

    @Override
    public void init() {
        readChannel(SUEDDEUTSCHE_ZEITUNG_URL);
    } // NewsReader()

    @Override
    public void start(Stage stage) {
        BorderPane topLevel = new BorderPane();
        topLevel.getStyleClass().add("border-pane");

        topLevel.setCenter(textFlow);

        MenuItemHandler handler = new MenuItemHandler(this);

        MenuBar menuBar = new MenuBar();
        Menu depthOfRecursion = new Menu("Channel");
        menuBar.getMenus().add(depthOfRecursion);

        MenuItem channel0 = createMenuItem(DEUTSCHE_WELLE_NAME,
                DEUTSCHE_WELLE_URL, handler);
        MenuItem channel1 = createMenuItem(GOETHE_INSTITUT_NAME,
                GOETHE_INSTITUT_URL, handler);
        MenuItem channel2 = createMenuItem(HELMHOLTZ_ASSOC_NAME,
                HELMHOLTZ_ASSOC_URL, handler);
        MenuItem channel3 = createMenuItem(SUEDDEUTSCHE_ZEITUNG_NAME,
                SUEDDEUTSCHE_ZEITUNG_URL, handler);

        depthOfRecursion.getItems().addAll(
                channel0,
                channel1,
                channel2,
                channel3);

        topLevel.setTop(menuBar);

        Scene scene = new Scene(topLevel, WIDTH, HEIGHT);
        scene.getStylesheets().add(NewsReader.class
                .getResource("NewsReader.css")
                .toExternalForm());

        stage.setTitle("Nachrichten aus Deutschland");
        stage.setScene(scene);
        stage.show();
    } // start( Stage )

    private MenuItem createMenuItem(String name,
            String url, MenuItemHandler handler) {
        MenuItem result = new MenuItem(name);
        result.setId(url);
        result.setOnAction(handler);
        return result;
    } // createMenuItem( String )

    public void readChannel(String url) {
        boolean ok = false;
        try {
            URL feedUrl = new URL(url);

            SyndFeedInput input = new SyndFeedInput();
            XmlReader reader = new XmlReader(feedUrl);
            SyndFeed feed = input.build(reader);

            this.stories.clear();
            for (Object o : feed.getEntries()) {
                if (o instanceof SyndEntryImpl) {
                    SyndEntryImpl e = (SyndEntryImpl) o;
                    
                    String title = e.getTitle();
                    String description = e.getDescription().getValue();
                    String link = e.getLink();
                    
                    Story story = new Story( title, description, link);
                    
                    this.stories.add(story);
                } // if
            } // for

            ok = true;
        } // try
        catch (Exception ex) {
            ex.printStackTrace();
            System.out.println("ERROR: " + ex.getMessage());
        } // catch( Exception )

        if (!ok) {
            System.out.println();
            System.out.println("FeedReader reads and prints "
                    + "any RSS/Atom feed type.");
            System.out.println("The first parameter must be "
                    + "the URL of the feed to read.");
            System.out.println();
        } // if

        this.textFlow.getChildren().clear();
        for (Story story : this.stories) {
            Text title = new Text(story.getTitle() + "\n");
            Text description = new Text(story.getDescription() + "\n\n");
            this.textFlow.getChildren().addAll( title, description );
        } // for
    } // readChannel( String )

    public static void main(String[] args) {
        launch(args);
    } // main( String [] )

} // NewsReader
